/********************************************/
/*-----------fitness.c----------------------*/
/********************************************/
/* For use with dsane.c, Eric R. Weeks & John M Burgess 
*  weeks@chaos.ph.utexas.edu, jburgess@chaos.ph.utexas.edu
*
* version 1.0: April 8, 1997
*
*   HOW THIS PROGRAM WORKS:  The variable "x" is the variable that
*   the input layer to the neural network will see.  The previous values
*   of this variable are kept track of in the xlag[] array.  The other
*   variables that the map may use are hidden from the neural network,
*   thus they are not stored in a lag[] array.  Only the function
*   apply_action() cares about these hidden variables.  The hidden
*   variables can be stored in the extravars[] array.
*
*   LOGISTIC MAP:
*     no additional variables
*
*   HENON MAP:
*     extravars[0]      is the additional variable
*
*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "params.h"
#include "dsane.h"

/* how many iterations we need to stay near the fixed point in order
 * to calculate lambda:   */
#define LYATHRESH 2

#define BONUS 1000.0
#define EPS 0.000001

#define MAXSAVE 20000		/* how many iterations can we save */
#define NUMVARS 10		/* how many extra variables are allowed */

#define MAXLAG 5		/* size of lag arrays */
float xlag[MAXLAG];      /* xlag[0] = current value, xlag[1] = previous */
float plag[MAXLAG];

static float size=-9.0;


/* if oldfitness > 0 then update dynamics_file and lambda_file */

float find_fitness(network *net,int max_steps,float oldfitness)
{
    float infx,supx;                        /* for calculating size:      */

    float bestlya,lya,lambda;               /* for finding lambda         */
    int nearfixpt,lyanum,bestlyanum;

    int bigperturbcount;		    /* for F_3 part of fit. func. */

    /* other variables */
    static float bestfit = -9e30;
    float avedelta;
    float x,x_next;
    int steps,i,j,numcnt;
    float fitness_temp,delta;
    extern float deltapmax[2];
    extern int period;
    extern int printsteps;
    extern int generation;
    extern float parameter;
    float deltalast = 0.0;
    float deltaP[2];
    float xsave[MAXSAVE],dpsave[MAXSAVE][2];
    FILE *best_file;
    extern FILE *lambda_file,*dynamics_file;
    float extravars[NUMVARS];

    lya = 1.0; lyanum = 0; nearfixpt = 0;
    bestlya = 1.0; bestlyanum = 0;
    fitness_temp = BONUS;
    bigperturbcount = 0;
    avedelta = 0.0; numcnt = 0;

    /* -------------------- set up start state --------------- */
    for (i=0;i<MAXLAG;i++)  plag[i] = 0.0;
    setup_initial_state(&x,extravars,deltaP);

    if (size<0.0)  {
	/* ---------------- figure out 'size' ----------------- */
	supx = -999.0;  infx = 999.0;
	x_next = x;
	/* first get rid of transients */
	for (j=0;j<30;j++)
	    x_next = apply_action(x_next,extravars,parameter,deltaP);
	/* now find size */
	for (j=0;j<50;j++)  {
	    x_next = apply_action(x_next,extravars,parameter,deltaP);
	    if (x_next<infx)  infx = x_next;
	    if (x_next>supx)  supx = x_next;
	}
	size = (supx-infx);
	/* fprintf(stderr,"SIZE %f   supx %f  infx %f\n",size,supx,infx); */
    }




    /*--- Iterate through the action-learn loop. ---*/
    for (steps=1;steps<=max_steps;steps++)  {
	setup_input(net,xlag,plag);
	activate_net(net);   
	get_deltaP(net,deltaP,deltapmax);
	if (steps<30)  {
	    /* further transient removal... */
	    deltaP[0]=0.0;  deltaP[1]=0.0;
	}

	if (steps<MAXSAVE)  {
	    xsave[steps] = x;
	    dpsave[steps][0] = deltaP[0];
	    dpsave[steps][1] = deltaP[1];
	}

	/*--- Apply action to the system / iterate map ---*/
	x = apply_action(x,extravars,parameter,deltaP);

	delta = fabs(xlag[0] - xlag[period]);
	if ((max_steps-steps<40))  {
	    avedelta += delta;
	    numcnt++;
	}
	if (steps>1)  {
	    if (nearfixpt==1)  {
		/* EPS in line below prevents divide by zero */
		if (deltalast>0.0)  lya *= (delta+EPS)/(deltalast+EPS);
		lyanum++;
		if ((deltalast>(0.1000*size)))  {
		    if (lyanum>LYATHRESH)  {
			/* store the information */
			bestlyanum += lyanum; bestlya *= lya;
		    }
		    nearfixpt = 0; lyanum = 0; lya = 1.0;
		}
	    } else if ((nearfixpt==0) && (deltalast<(0.01000*size)))  {
		nearfixpt = 1;
	     }
	}
	deltalast = delta;

	if ((fabs(deltaP[0])>(deltapmax[0]*.95))||
		(fabs(deltaP[1])>(deltapmax[1]*.95)))
	    bigperturbcount++;
    }


    /* ----------------- determine fitness value ------------- */

    if (numcnt>0)  {
	/* this is "F_1" in the paper: should be positive */
	avedelta = (avedelta/((float)numcnt))/(.95*size);
	fitness_temp += (AAA)*(1.0-avedelta);
    }
    if (lyanum>LYATHRESH)  {
	bestlyanum += lyanum;
	bestlya *= lya;
    }
    if (bestlyanum>0)  {
	/* this is "F_2" in the paper: can be negative if lambda is huge. */
	lambda = (pow(bestlya,1.0/((float)bestlyanum)));
	if (lambda<0.0001)  lambda = 0.0001;
	fitness_temp += (BBB)*(1.0 - log(lambda+0.0001));
	if (oldfitness>0.0)  {
	    fprintf(lambda_file,"%4d ",generation);
	    fprintf(lambda_file," %4d %f\n",bestlyanum,lambda);
	    fflush(lambda_file);
	}
    } else {
	/* this is "F_3" in the paper: always a positive reward */
	fitness_temp+=
	    CCC*((float)(max_steps-bigperturbcount))/((float)max_steps);
    }

    /* ------------ if fantastic network, save record of dynamics ----- */
    if (fitness_temp>bestfit)  {
	bestfit = fitness_temp + 1.0;
	best_file=openfile("best","w");
	for (steps=1;steps<=max_steps;steps++)  {
	    if (steps<MAXSAVE)
		fprintf(best_file,"%5d %9f %9f %9f\n",steps,
			    xsave[steps],dpsave[steps][0],dpsave[steps][1]);
	}
	fclose(best_file);
    }

    /* ----- if best net of this generation, save record of dynamics ----- */
    if ((oldfitness>0)&&(printsteps>0))  {
	for (steps=max_steps-printsteps;steps<=max_steps;steps++)  {
	    fprintf(dynamics_file,"gen %d ",generation);
	    fprintf(dynamics_file,"fit %5.0f %5.0f ",oldfitness,fitness_temp);
	    fprintf(dynamics_file,"(n,x,dp1,dp2): ");
	    fprintf(dynamics_file,"%5d %9f %9f %9f\n",steps,
			xsave[steps],dpsave[steps][0],dpsave[steps][1]);
	}
	fflush(dynamics_file);
    }
 
    return fitness_temp;
} 





/** apply_action walks the system's dynamics one iteration 
    This fcn will change the value of extravars[] as well. **/

float apply_action(float x, float extravars[],float P,float deltaP[])
{ 
    extern float noiselevel;
    extern int themap;	/* 1 = logistic   2 = henon   3 = usermap */
    int i;
    float x_next;

    if (themap==HENON)  {
	/* ---------- HENON MAP ---------- */
	x_next=(P + deltaP[0])+((0.3+deltaP[1]) *(extravars[0])) - (x * x);
	extravars[0] = x;	/* this is 'y' */
	x_next += (drand48()-0.5)*noiselevel;
	if (x_next < (-1.78))  x_next = -1.78;
	if (x_next > 1.78)  x_next = 1.78;
    } else if (themap==LOGISTIC)  {
	/* ---------- LOGISTIC MAP ---------- */
	x_next = x * (P + deltaP[0]) * (1.0 - x);
	x_next += (drand48()-0.5)*noiselevel;
	if (x_next<0.0)  x_next = 0.0;
	if (x_next>1.0)  x_next = 1.0;
    } else {
	/* ---------- USER MAP --------------- */
	x_next = x;
	usermap(&x_next,extravars,deltaP,P,noiselevel);
    }

    /* UPDATE LAGGED VARIABLES */
    for (i=(MAXLAG-1);i>0;i--)  {
	xlag[i] = xlag[i-1];
	plag[i] = plag[i-1];
    }
    /* Note "plag" array only applies to deltaP[0], currently */
    plag[0] = deltaP[0];
    xlag[0] = x_next;
    
    return x_next;
}




void neteval(network* net)
{
    float extravars[NUMVARS];
    int test,steps,j,min,max;
    extern int generation;
    extern FILE *control_file;
    int flag,controlby,count,num;
    float average,delta,x_next;
    extern float deltapmax[2];
    extern int period;
    extern float parameter;
    float deltaP[2];

    /* do 200 tests */
    average = 0.0;
    num = 0;
    count = 200;
    max = -1;min=9999;
    for (test=0;test<count;test++)  {
	setup_initial_state(&x_next,extravars,deltaP);
	for (j=0;j<30;j++)  
	    x_next = apply_action(x_next,extravars,parameter,deltaP);

	/* do one test */
	flag = 0; controlby = 0;
	for (steps=1;steps<=3000;steps++)  {
	    setup_input(net,xlag,plag);
	    activate_net(net);   
	    get_deltaP(net,deltaP,deltapmax);
	    x_next = apply_action(x_next,extravars,parameter,deltaP);
	    delta = fabs(xlag[0] - xlag[period]);
	    if (flag==0)  {
		if (delta<size*0.01000)  {
		    flag=1; controlby=steps;
		}
	    } else {
		if (delta>size*0.1000)  {
		    flag=0; controlby=0;
		}
	    }
	}
	if (controlby>0)  {
	    average += controlby;
	    num++;
	    if (controlby<min)  min=controlby;
	    if (controlby>max)  max=controlby;
	}
    }
    if (num>0)  {
	fprintf(control_file,"generation %5d : ave iterations to ctrl %.1f",
		generation,average/((float)num));
    } else {
	fprintf(control_file,"generation %5d : never controlled",
		generation);
    }
    fprintf(control_file," fail %d/%d min %d max %d\n",count-num,count,min,max);
    fflush(control_file);
}





void setup_initial_state(float *x, float variables[],float deltaP[])
{
    extern int themap;	/* 1 = logistic   2 = henon   3 = usermap */
    deltaP[0] = 0; deltaP[1] = 0;
    if (themap==LOGISTIC)  {
	(*x) = .1+.8*((float)drand48());
    } else if (themap==HENON)  {
	(*x) = .1+.8*((float)drand48());
	variables[0] = (*x);
    } else {
	initialize_usermap(x, variables);
    }
}
