/*** dsane.c
    Eric R. Weeks, John M. Burgess
    http://chaos.ph.utexas.edu/~weeks/dsane/


    portions taken from "sane.c" originally written by:
    David E. Moriarty, June 1994

version 1.0: April 8, 1997.
     
    revision: Eric Weeks and John Burgess -- Fall 1995 thru Spring 1997
    (weeks@chaos.ph.utexas.edu, jburgess@chaos.ph.utexas.edu)

This program works with the following files:
   main.c
   fitness.c
   usermap.c
   network.c
   sane.h
   params.h

This file contains the following functions:

initialize_dsane():	initialize variables, open files
create_pop():		create a population of neurons
evolve():		evolve a population of neurons
 qsort_neurons():	for ranking according to fitness
  qpartition_neurons():	used by qsort_neurons()
eval_pop():		find the fitness of each neuron
 getfitness():		find the fitness of a network
 mutate():		mutate a group of genes
 crossover():		crossover several pairs of genes
activate_net():		evaluate a neural network: 
                                called by find_fitness(in fitness.c)
printoutwts():		subroutine for print_pop, print_network
print_pop():		print out entire population
print_network():	print out a given network
exit_program():		stuff to do when evolution is finished

THIS FILE IS FORMATTED FOR A TAB STOP OF 8

Explanation of how directed sane is implemented:
    At the start of a generation, each gene gets a unique new 
id number ("newidnum") and gets "idnum" from its parents newidnum. 
When networks are formed during that generation, the top "netdefnnum" 
network definitions are saved -- which includes their fitness and 
the "newidnum"'s of the genes that formed them.  
    Networks are formed where the stored id numbers in the
definition match the "idnum" of a given gene.  There are usually 
several possible choices as there are many offspring of a good
neuron, all with the "idnum" of their parent (but with distinct
"newidnum" so that we can create new network definitions).
    Note the big differences between Directed SANE and Moriarty's
Hierarchical SANE.  Our network definitions have a short lifetime; 
they are recreated each generation.  The objects pointed at by the 
network definitions are non-unique; they are all the neurons of a
given type, rather than a specific neuron.

    For more explanation, see our paper.

    ***/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include "params.h"
#include "dsane.h"


/*Global variables**************************************/
defnnet olddefn[MAXNET];	/* previous generation network defns */
defnnet newdefn[MAXNET];	/* current generation best networks so far */
int generation;                 /* needed for "evolve", and for operators */
int opernum[OPNUM];		/* how many of each operator to use */
FILE *dynamics_file,*evolution_file,*lambda_file,*control_file;
int fourtimeswts=4*(NUM_INPUTS+NUM_OUTPUTS);


int initialize_dsane(float fracdefnuse,int numtrials)
{
    int i,j;
    int preserve;
    extern int netdefnnum,maxdefnuse,printsteps;

    evolution_file = openfile("evol", "w");
    lambda_file = openfile("lambda", "w");
    control_file = openfile("control", "w");
    if (printsteps>0)
	dynamics_file = openfile("dyn", "w");
    /* lambda_file, dynamics_file, and control_file are written by fitness.c */

    maxdefnuse = ((int)(0.01*((float)numtrials)*fracdefnuse));
    if (maxdefnuse > numtrials)  maxdefnuse = numtrials;
    /* POP_SIZE is size of population */
    /* PERCENT_MUTATE is fraction of population replaced with mutants
     * PERCENT_CROSSOVER is fraction of pop replaced with crossover
     * thus, "percent_preserved" = 100% - PERCENT_MUTATE - PERCENT_CROSSOVER
     */
    /* crossover = #0 */
    opernum[0] = ((int)( 0.01*PERCENT_CROSSOVER*POP_SIZE ));
    /* four levels of mutation */
    opernum[1] = ((int)( 0.01*PERCENT_MUTATE*POP_SIZE*0.25 ));
    opernum[2] = ((int)( 0.01*PERCENT_MUTATE*POP_SIZE*0.25 ));
    opernum[3] = ((int)( 0.01*PERCENT_MUTATE*POP_SIZE*0.25 ));
    opernum[4] = ((int)( 0.01*PERCENT_MUTATE*POP_SIZE*0.25 ));
    preserve = POP_SIZE;
    for (i=0;i<OPNUM;i++)  preserve -= opernum[i];

    if (preserve<1)  {
	fprintf(stderr,"ERROR: PERCENT_CROSSOVER+PERCENT_MUTATE is larger\n");
	fprintf(stderr,"than 100.\n");
	exit(1);
    }

    /* initialize network definitions here */
    for (i=0;i<netdefnnum;i++)  {
	for (j=0;j<ZETA;j++) 
	    olddefn[i].idnet[j] = -1;
	olddefn[i].fitness = -9.0;
    }
    return preserve;
}



/** create_pop creates a random population of neurons
 **
 ** Also, this initializes the network pointers array
 **/

void create_pop(newneuron *new_pop[])
{
    int i,j;
    for (i=0;i<POP_SIZE;++i)  {
	for(j=0;j<(fourtimeswts);++j)  {
	    /* next line used to be lrand48() rather than j */
	    new_pop[i]->label[j] = j%((NUM_INPUTS)+(NUM_OUTPUTS));
	    if (new_pop[i]->label[j] >= (NUM_OUTPUTS))
		new_pop[i]->label[j]+=128;
	    new_pop[i]->weight[j] = (-2.0+4.0*drand48());
	    if (lrand48()%2==0)
		new_pop[i]->weight[j] *= -1.0;
	}
	decode_gene(new_pop[i]);
	new_pop[i]->neuron.sum = 0.0;
	new_pop[i]->neuron.sigout = 0.0;
	new_pop[i]->fitness = 0.0;
	new_pop[i]->method = 0;	    /* creation here */
	new_pop[i]->birth = -1;	    /* born at beginning */
	new_pop[i]->idnum = i;	    /* I am neuron #i ! */
	new_pop[i]->newidnum = i;
    }

}


void decode_gene(newneuron *gene)
{
    int j,k,label;
    float weight;
    int from,to;

    for (j=0;j<NUM_INPUTS;j++)  gene->neuron.weight[j] = 0.0;
    for (j=0;j<NUM_OUTPUTS;j++)  gene->neuron.outwt[j] = 0.0;

    for (k=0;k<fourtimeswts;k++)  {
        label = gene->label[k];
        weight = gene->weight[k]; 
        if (label>=128) {    /*connection from input to hidden layer*/
            from = label % NUM_INPUTS;
            /*comment: input unit = label MOD NUM_INPUTS*/
            gene->neuron.weight[from] += weight;
        } else {             /*connection from hidden to output layer*/
            to = label % NUM_OUTPUTS;
            gene->neuron.outwt[to] += weight;
        }
    }
}



/** evolve is the main genetic function.  The population is first
 evaluated in the given task.  evolve then ranks the neurons and
 then uses any and all operators to produce new neurons from the best
 of the current generation.
    **/

void evolve(newneuron *pop[],int cycles,int preserve,float stopfit)
/*
newneuron *pop[];  /* a population of neurons 
int cycles;        /* the number of generations to evolve
int preserve;      /* how many neurons are unchanged each generation 
float stopfit;     /* stop evolution when a network is above this fitness
*/
{
    register int j,k;
    float mult;

    mult = 1.0;
    /* cycle population */
    for (generation=0;generation<cycles;++generation) {
	eval_pop(pop,generation,stopfit);	/*evaluate neurons*/
	qsort_neurons(pop,0,POP_SIZE-1); /*sort neurons*/

	/* if a neuron is unreplaced, it retains its idnum */
	for (j=0;j<POP_SIZE;j++)  pop[j]->idnum = pop[j]->newidnum;


	k = POP_SIZE-1;    /* index of neuron to replace */
	k = crossover(pop,preserve,k,opernum[0]);
	k = mutate(pop,preserve,k,mult*1.0,opernum[1],1);
	k = mutate(pop,preserve,k,mult*0.1,opernum[2],2);
	k = mutate(pop,preserve,k,mult*0.01,opernum[3],3);
	k = mutate(pop,preserve,k,mult*0.001,opernum[4],4);

	/* only neurons born right now: */
	for (j=k+1;j<POP_SIZE;j++)   pop[j]->birth = generation;

	/* everyone gets a new name: */
	for (j=0;j<POP_SIZE;j++)   pop[j]->newidnum = j;

    }
    fprintf(stdout,"program finished, never reached goal.\n");
}


int mutate(newneuron *pop[],int max,int replace,float level,int number,int methodused)
/*
newneuron *pop[];    a population of neurons
int max;	     how many neurons to mutate 
int replace;	     which neuron to replace - starting 
float level;	     how much mutation to do 
int number;	     how many neurons to mutate 
int methodused;      which operator is this 
*/
{
    int ii,j,i,k,n;

    k = replace;
    for (ii=0;ii<number;ii++)  {
	i = (int)(drand48()*max);
        for(j=0;j<fourtimeswts;++j)  {
            pop[k]->label[j] = pop[i]->label[j];
            pop[k]->weight[j] = pop[i]->weight[j];
            pop[k]->weight[j] *= exp(level*(drand48()-0.5));
            if (drand48()<.02*level)  pop[k]->weight[j] *= -1.0;
        }
	decode_gene(pop[k]);
	pop[k]->method = methodused;
	pop[k]->idnum = pop[i]->newidnum;
	k--;
    }
    return k;
}




int crossover(newneuron *pop[],int max,int replace,int number)
/*
newneuron *pop[];  a population of neurons 
int max;	   how many neurons to choose from for parents 
int replace;	   which neuron to replace - starting 
int number;        how many new neurons to create 
*/
{
    int j,k,ii,n;
    int p1,p2;    /* parent indices */

    k = replace;
    for (ii=0;ii<number;ii++)  {
	p1 = (int)(drand48()*max);
	p2 = p1;
	while (p2==p1)  p2 = (int)(drand48()*number);
	/* now we have two unique parents */
	for (j=0;j<NUM_INPUTS;j++)  {
	    if (drand48()<.5)  {
		pop[k]->neuron.weight[j] = pop[p1]->neuron.weight[j];
		for (n=0;n<fourtimeswts;n++)  {
		    if ((pop[p1]->label[n] % NUM_INPUTS)==j &&
                              (pop[p1]->label[n]>=128))  {
			pop[k]->label[n] = pop[p1]->label[n];
			pop[k]->weight[n] = pop[p1]->weight[n];
		    }
		}
	    } else {
		pop[k]->neuron.weight[j] = pop[p2]->neuron.weight[j];
		for (n=0;n<fourtimeswts;n++)  {
		    if ((pop[p2]->label[n] % NUM_INPUTS)==j &&
                              (pop[p2]->label[n]>=128))  {
			pop[k]->label[n] = pop[p2]->label[n];
			pop[k]->weight[n] = pop[p2]->weight[n];
		    }
		}
	    }
	}
	for (j=0;j<NUM_OUTPUTS;j++)  {
	    if (drand48()<.5)  {
		pop[k]->neuron.outwt[j] = pop[p1]->neuron.outwt[j];
		for (n=0;n<fourtimeswts;n++)  {
		    if ((pop[p1]->label[n] % NUM_OUTPUTS)==j &&
                              (pop[p1]->label[n]<128))  {
			pop[k]->label[n] = pop[p1]->label[n];
			pop[k]->weight[n] = pop[p1]->weight[n];
		    }
		}
	    } else {
		pop[k]->neuron.outwt[j] = pop[p2]->neuron.outwt[j];
		for (n=0;n<fourtimeswts;n++)  {
		    if ((pop[p2]->label[n] % NUM_OUTPUTS)==j &&
                              (pop[p2]->label[n]<128))  {
			pop[k]->label[n] = pop[p2]->label[n];
			pop[k]->weight[n] = pop[p2]->weight[n];
		    }
		}
	    }
	}
	pop[k]->method = 6;
	pop[k]->idnum = pop[p1]->newidnum;    
        /* save info that this neuron came from parent p1: */
	/* p1 and p2 are chosen randomly; only info about  */
	/* one parent is saved                             */
	
	k--;
    }
    return k;
}



/** quick sort the neurons.  Use for large populations (>50) **/

void qsort_neurons(newneuron *pop[],int p,int r)
{
    int q;
    if (p<r) {
	q = qpartition_neurons(pop,p,r);
	qsort_neurons(pop,p,q);
	qsort_neurons(pop,q+1,r);
    }
}


/** partition function for qsort.  **/

int qpartition_neurons(newneuron *pop[],int p,int r)
{
  register int i,j;
  newneuron *x,*temp;

  x = pop[p];
  i = p - 1;
  j = r + 1;
  while(1) {
    do{
      --j;
    }while(pop[j]->fitness < x->fitness);
    do{
      ++i;
    }while(pop[i]->fitness > x->fitness);
    if (i < j) {
      temp = pop[i];
      pop[i] = pop[j];
      pop[j] = temp;
    } else {
      return j;
    }
  }
}



/** eval_pop evaluates entire population to get the fitness of
  each neuron.  This function is called by evolve and should put the
  fitness of each neuron in the neuron's fitness field
  (pop[i]->fitness).
    **/

void eval_pop(newneuron *pop[],int generation, float stopfit)
{
    int i,j,k,m,n;
    float fitness,tempfit;
    network net;
    network bestnet;
    newneuron *net_pop[ZETA];
    int idnumber[ZETA],id;
    int worstid;
    float worstfit,bestdefnfit = -9e30;
    float fitttl = 0.0,best = -9e9;
    /* next six variables to evaluate the effectiveness of directed sane */
    int better;
    int direcgood=0,nondirecgood=0;
    extern int numtrials,netdefnnum,maxdefnuse,numsteps;

    /* ----------- initialize genes ----------------- */
    for(i=0;i<POP_SIZE;++i)  {
	pop[i]->fitness = 0.0;
	pop[i]->tests = 0;
    }

    /* -------------- inititalize network definitions ------------*/
    for (i=0;i<netdefnnum;i++)  {
	newdefn[i].fitness = -999.9;
	for (j=0;j<ZETA;j++)  newdefn[i].idnet[j] = -1;
	if (olddefn[i].fitness>bestdefnfit)
	    bestdefnfit = olddefn[i].fitness;
    }

    /*EVALUATION STAGE-- Create numtrials networks, each containing
    ZETA neurons selected randomly from the population.  Each
    network is evaluated and each participating neuron receives the
    fitness evaluations of each network it parcipates in.*/

    for(i=0;i<numtrials;++i) {
	/* ---------- CREATE A NETWORK ---------- */
	if (i<maxdefnuse)  {
	    /* k is choice of network defn to use */
	    k = (i % netdefnnum);
	}
	for (j=0;j<ZETA;j++)  {
	    /* find random default subpopulation */
	    net_pop[j] = pop[lrand48()%POP_SIZE];
	    if (i<=maxdefnuse)  {
		/* use Directed SANE to form network */
		id = olddefn[k].idnet[j];
		n = 0;    /* number of matches found */
		/* the m loop does a 1-pass random choice of all
		   neurons with idnum==id by rand comparison with 1/n */
		for (m = 0;m<POP_SIZE;m++)
		    if (pop[m]->idnum == id)  {
			n++;
			/* found a match -- do we use? */
			if (drand48()<(1.0/((double)(n))))
			    net_pop[j] = pop[m];
			if (pop[m]->birth < (generation-1))
			    if (k==i)  {
				/* first netdefnnum trials, use
				 * exact same neurons as before */
				net_pop[j] = pop[m];
				m = POP_SIZE+999; /* exit m loop */
			    }
		    }
		/* if n==0 then use random default given above! */
	    }
	    ++net_pop[j]->tests;
	    idnumber[j] = net_pop[j]->newidnum;
	}
	/* transform neurons in population into hidden unit neurons */
	for(j=0;j<ZETA;j++)  net.hidden_unit[j] = net_pop[j]->neuron;
	/* ---------- NETWORK IS CREATED ---------- */

	/* ---------- EVALUATE NETWORK ---------- */
	fitness = 0.0;
	getfitness(&net,&fitness,numsteps);
	if (i<maxdefnuse)  {
	    if (fitness>bestdefnfit)  direcgood++;
	} else {
	    if (fitness>bestdefnfit)  nondirecgood++;
	}

	/* ---------- FITNESS COMPARED WITH REST OF POPULATION ---------- */
	if (fitness > best)  {
	    bestnet = net;
	    best = fitness;
	    better = ( (i<maxdefnuse) ? 1 : 0 );
	}
	fitttl += fitness;

	/* --------- FIND WORST NETWORK DEFN & PERHAPS REPLACE ----- */
	worstid = -1;
	worstfit = fitness;
	for (j=0;j<maxdefnuse;j++)
	    if (newdefn[j].fitness < worstfit)  {
		worstid = j;
		worstfit = newdefn[j].fitness;
	    }
	/* if current defn is better than worst then replace: */
	if (worstid>-1)  {
	    for (j=0;j<ZETA;j++) 
		newdefn[worstid].idnet[j] = idnumber[j];
	    newdefn[worstid].fitness = fitness;
	}


	/* ---------- UPDATE FITNESS OF GENE ---------- */
	for(j=0;j<ZETA;j++)  {
	    if (fitness > (net_pop[j]->fitness))
	    net_pop[j]->fitness = fitness;
	}
    }
    /* ------------- DONE WITH NETWORK EVALUATIONS ----------- */

    for (j=0;j<maxdefnuse;j++)  olddefn[j] = newdefn[j];
    print_network(&bestnet,best);
    (void)find_fitness(&bestnet,numsteps,best);
    if (generation % 10 == 0)  neteval(&bestnet);


    /* ----------------- OUTPUT ----------------- */
    fprintf(evolution_file,"%4d ",generation);
    fprintf(evolution_file,"bestnet %5.0f ",best);
    fprintf(evolution_file,"ave %5.0f ",(fitttl/(double)(numtrials)));
    if (netdefnnum>0)
        if (better==1)  {
	    fprintf(evolution_file,"D %2d o %2d  ",direcgood,nondirecgood);
        } else {
	    fprintf(evolution_file,"d %2d O %2d  ",direcgood,nondirecgood);
	}
    fprintf(evolution_file,"\n");
    fflush(evolution_file);

    /* ---------- SHOULD WE STOP THE PROGRAM NOW ? ---------- */
    if (best>stopfit)  {
	/* check to make sure this is really good */
	tempfit = 0.0;
	for (j=0;j<10;j++)  {
	    fitness = find_fitness(&bestnet,numsteps,NOPRINT);
	    tempfit += fitness;
	    if (fitness<stopfit)  j = 999;
	}
	if (fitness>stopfit)  {
	    /* OK, it's good! let's quit now */
	    fprintf(stdout,"dsane finished at generation %d\n",generation);
	    exit_program(pop,&bestnet);
	}
    }
}

void getfitness(network *net_ptr,float *fitness,int numsteps)
{
    int m;
    float temp_fitness;
    extern int doublecheck;

    temp_fitness = find_fitness(net_ptr,numsteps,NOPRINT);
    (*fitness) += temp_fitness;
    if (doublecheck>1)  {
	for (m=1;m<doublecheck;m++)  {
	    temp_fitness = find_fitness(net_ptr,numsteps,NOPRINT);
	    (*fitness) += temp_fitness;
	}
	(*fitness) /= doublecheck;
    }
}




/** activate_net will activate the neural network.  The input layer
    must be setup before calling activate_net.
    This function is called from find_fitness() in fitness.c
    **/

void activate_net(network *net)
{
   register int i,j;
   float sum;

   /* evaluate hidden layer */
    for (i=0;i<ZETA;++i) {  /*for each hidden unit*/
	sum = 0.0;
	/*for each connection into the hidden unit, sum the weights of
	the active input units. */
	for (j=0;j<NUM_INPUTS;++j) {
	    sum += (net->hidden_unit[i].weight[j]) * (net->input_unit[j]);
	}
	net->hidden_unit[i].sigout =   1.0/(1.0+exp(-sum));
    }

    /*evaluate output layer*/

    for(i=0;i<NUM_OUTPUTS;++i) {
	sum = 0.0;
	for (j=0;j<ZETA;++j)
	    sum+=net->hidden_unit[j].sigout * net->hidden_unit[j].outwt[i];
	net->output_unit[i].sigout = 1.0/(1.0+exp(-sum)); 
    }
}



void printoutwts(node *neuron,FILE *filename)
{
    int j;

    fprintf(filename,"inputs: ");
    for (j=0;j<NUM_INPUTS;j++)  {
	fprintf(filename,"%10.2e\t",neuron->weight[j]);
    }
    fprintf(filename,"\n");
    fprintf(filename,"outputs: ");
    for (j=0;j<NUM_OUTPUTS;j++)  {
	fprintf(filename,"%10.2e\t",neuron->outwt[j]);
    }
    fprintf(filename,"\n");
}



/* prints out the entire population of neurons */
void print_pop(newneuron *pop[])
{
    int i;
    node neuron;
    FILE *population_file;

    population_file=openfile("popul","w");

    for (i=0;i<POP_SIZE;i++)  {
	neuron = pop[i]->neuron;
	fprintf(population_file,"fitness: %.0f  ",pop[i]->fitness);
	fprintf(population_file,"born: %d  ",pop[i]->birth);
	fprintf(population_file,"method used: ");
	if (pop[i]->method==6)  {
	    fprintf(population_file,"crossover");
	} else {
	    fprintf(population_file,"mutation ");
	}
	fprintf(population_file,"\n");
	printoutwts(&neuron,population_file);
    }
    fclose(population_file);

}



/* print out connections in a network */
void print_network(network* net,float fitness)
{
    int i;
    FILE *network_file;

    network_file = openfile("net","w");
    fprintf(network_file,"%d input neurons\n",NUM_INPUTS);
    fprintf(network_file,"%d hidden neurons\n",ZETA);
    fprintf(network_file,"%d output neurons\n",NUM_OUTPUTS);
    for (i=0;i<ZETA;i++)  {
	printoutwts(&net->hidden_unit[i],network_file);
    }
    fflush(network_file);
    fclose(network_file);
}





void exit_program(newneuron *pop[], network *bestnet)
{
    extern int printsteps;
    qsort_neurons(pop,0,POP_SIZE-1); /*sort neurons*/
    print_pop(pop);
    neteval(bestnet);
    if (printsteps>0)  {
	fflush(dynamics_file);
	fclose(dynamics_file);
    }
    fflush(evolution_file);
    fclose(evolution_file);
    fflush(lambda_file);
    fclose(lambda_file);
    fflush(control_file);
    fclose(control_file);
    exit(0);
}

