/*
 * Open Source Physics software is free software as described near the bottom of this code file.
 *
 * For additional information and documentation on Open Source Physics please see: 
 * <http://www.opensourcephysics.org/>
 */

package org.opensourcephysics.sip.ch09;
import org.opensourcephysics.numerics.Function;

/**
 * Analyze  Fourier sine and cosine coefficients.
 *
 * @author Wolfgang Christian, Jan Tobochnik, Harvey Gould
 * @version 1.0
 */
public class Analyze {
  Function f;
  double period, delta;
  double omega0;
  int N;

  /**
   * Constructs the Analyze object
   *
   * @param N the number of points
   * @param dt     the sampling interval
   */
  Analyze(Function f, int N, double delta) {
    this.f = f;
    this.delta = delta;
    this.N = N;
    period = N*delta;
    omega0 = 2*Math.PI/period;
  }

  /**
   * Gets a Fourier sine coefficient
   *
   * @param n coeff index
   *
   * @return the coefficient
   */
  double getSineCoefficient(int n) {
    double sum = 0;
    double t = 0;
    for(int i = 0;i<N;i++) {
      sum += f.evaluate(t)*Math.sin(n*omega0*t);
      t += delta;
    }
    return 2*delta*sum/period;
  }

  /**
   * Gets a Fourier cosine coefficient.
   *
   * @param n coeff index
   *
   * @return the coefficient
   */
  double getCosineCoefficient(int n) {
    double sum = 0;
    double t = 0;
    for(int i = 0;i<N;i++) {
      sum += f.evaluate(t)*Math.cos(n*omega0*t);
      t += delta;
    }
    return 2*delta*sum/period;
  }
}

/* 
 * Open Source Physics software is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License (GPL) as
 * published by the Free Software Foundation; either version 2 of the License,
 * or(at your option) any later version.

 * Code that uses any portion of the code in the org.opensourcephysics package
 * or any subpackage (subdirectory) of this package must must also be be released
 * under the GNU GPL license.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA
 * or view the license online at http://www.gnu.org/copyleft/gpl.html
 *
 * Copyright (c) 2007  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */
